# 📷 Fitur Upload Foto KTP - ENSIA

## ✅ **Fitur Baru: Upload Foto KTP untuk Data Pelanggan**

Form tambah data pelanggan sekarang dilengkapi dengan fitur upload foto KTP dengan validasi dan preview yang lengkap.

---

## 🎯 **Fitur Upload Foto KTP**

### **📁 Upload File:**
- **Format**: JPG, JPEG, PNG
- **Ukuran Maksimal**: 5MB
- **Nama File**: Otomatis dengan format `ktp_timestamp_nik.extension`
- **Lokasi**: `uploads/ktp/`

### **🔍 Validasi File:**
- **Tipe File**: Hanya image/jpeg, image/jpg, image/png
- **Ukuran File**: Maksimal 5MB
- **Validasi Client-side**: JavaScript real-time
- **Validasi Server-side**: PHP backend

### **👁️ Preview Foto:**
- **Real-time Preview**: Tampil langsung setelah pilih file
- **Thumbnail**: Ukuran 200x150px
- **Hapus Foto**: Tombol untuk menghapus sebelum upload
- **Drag & Drop**: Support drag and drop upload

---

## 🛠️ **Implementasi Teknis**

### **1. Database Update:**
```sql
-- Tambahkan kolom foto_ktp ke tabel customers
ALTER TABLE customers ADD COLUMN foto_ktp VARCHAR(255) DEFAULT NULL AFTER nik_ktp;

-- Tambahkan index untuk performa
CREATE INDEX idx_foto_ktp ON customers(foto_ktp);
```

### **2. PHP Upload Handler:**
```php
// Handle upload foto KTP
$foto_ktp = '';
if (isset($_FILES['foto_ktp']) && $_FILES['foto_ktp']['error'] == 0) {
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png'];
    $max_size = 5 * 1024 * 1024; // 5MB
    
    if (in_array($_FILES['foto_ktp']['type'], $allowed_types) && 
        $_FILES['foto_ktp']['size'] <= $max_size) {
        
        $file_extension = pathinfo($_FILES['foto_ktp']['name'], PATHINFO_EXTENSION);
        $foto_ktp = 'ktp_' . time() . '_' . $nik_ktp . '.' . $file_extension;
        $upload_path = 'uploads/ktp/' . $foto_ktp;
        
        if (!move_uploaded_file($_FILES['foto_ktp']['tmp_name'], $upload_path)) {
            $error = 'Gagal mengupload foto KTP!';
        }
    } else {
        $error = 'Format foto KTP tidak valid! Gunakan JPG/PNG maksimal 5MB.';
    }
}
```

### **3. HTML Form:**
```html
<form method="POST" enctype="multipart/form-data">
    <!-- Field upload foto KTP -->
    <div class="mb-4">
        <label for="foto_ktp" class="form-label">
            <i class="fas fa-camera me-2"></i>Foto KTP
        </label>
        <input type="file" class="form-control" id="foto_ktp" name="foto_ktp" 
               accept="image/jpeg,image/jpg,image/png">
        <div class="form-text">
            Format: JPG/PNG, maksimal 5MB. Foto akan otomatis diresize.
        </div>
        <div id="foto-preview" class="mt-2" style="display: none;">
            <img id="preview-img" src="" alt="Preview Foto KTP" 
                 class="img-thumbnail" style="max-width: 200px; max-height: 150px;">
            <button type="button" class="btn btn-sm btn-outline-danger ms-2" 
                    onclick="removeFoto()">
                <i class="fas fa-times me-1"></i>Hapus
            </button>
        </div>
    </div>
</form>
```

### **4. JavaScript Validasi:**
```javascript
// Preview foto KTP
document.getElementById('foto_ktp').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        // Validasi ukuran file (5MB)
        if (file.size > 5 * 1024 * 1024) {
            alert('Ukuran file terlalu besar! Maksimal 5MB.');
            this.value = '';
            return;
        }
        
        // Validasi tipe file
        const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png'];
        if (!allowedTypes.includes(file.type)) {
            alert('Format file tidak valid! Gunakan JPG atau PNG.');
            this.value = '';
            return;
        }
        
        // Tampilkan preview
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('preview-img').src = e.target.result;
            document.getElementById('foto-preview').style.display = 'block';
        };
        reader.readAsDataURL(file);
    }
});
```

---

## 📁 **Struktur File**

### **Folder Upload:**
```
uploads/
└── ktp/
    ├── ktp_1694067123_1234567890123456.jpg
    ├── ktp_1694067124_1234567890123457.png
    └── ...
```

### **Nama File:**
- **Format**: `ktp_timestamp_nik.extension`
- **Contoh**: `ktp_1694067123_1234567890123456.jpg`
- **Timestamp**: Unix timestamp untuk unik
- **NIK**: NIK KTP pelanggan
- **Extension**: Sesuai file asli (jpg, png)

---

## 🔒 **Keamanan**

### **Validasi File:**
- ✅ **Tipe MIME**: Validasi tipe file yang benar
- ✅ **Ekstensi**: Hanya .jpg, .jpeg, .png
- ✅ **Ukuran**: Maksimal 5MB
- ✅ **Nama File**: Sanitized dan unik

### **Upload Security:**
- ✅ **Path Traversal**: Dicegah dengan validasi path
- ✅ **File Overwrite**: Dicegah dengan nama unik
- ✅ **Directory**: File disimpan di folder khusus
- ✅ **Permission**: Folder upload dengan permission 755

---

## 🎨 **UI/UX Features**

### **Upload Area:**
- **Drag & Drop**: Support drag and drop
- **Click to Upload**: Klik area untuk pilih file
- **Visual Feedback**: Hover dan drag over effects
- **Icon**: Font Awesome camera icon

### **Preview:**
- **Real-time**: Preview langsung setelah pilih
- **Thumbnail**: Ukuran yang sesuai (200x150px)
- **Remove Button**: Tombol hapus dengan icon
- **Responsive**: Tampil baik di semua device

### **Validasi:**
- **Real-time**: Validasi saat pilih file
- **Error Messages**: Pesan error yang jelas
- **File Info**: Informasi format dan ukuran
- **Success Feedback**: Konfirmasi berhasil

---

## 🧪 **Test & Demo**

### **URL Demo:**
- **Demo Form**: `http://localhost:8080/demo_add_customer.php`
- **Main Form**: `http://localhost:8080/add_customer.php` (login required)

### **Fitur Demo:**
- Form lengkap dengan data sample
- Upload area dengan drag & drop
- Preview foto real-time
- Validasi client-side
- Simulasi submit

---

## 📊 **Data Sample**

### **Form Fields:**
- **Nama Lengkap**: Ahmad Wijaya
- **Pekerjaan**: Karyawan Swasta
- **Alamat**: Jl. Merdeka No. 123, Kelurahan Cikole, Kecamatan Cikole, Kota Sukabumi
- **Nomor HP**: 081234567890
- **Email**: ahmad.wijaya@email.com
- **NIK KTP**: 1234567890123456
- **Foto KTP**: Upload area dengan preview

---

## 🚀 **Cara Menggunakan**

### **1. Upload Foto:**
1. Klik area upload atau drag & drop file
2. Pilih file JPG/PNG maksimal 5MB
3. Lihat preview foto yang muncul
4. Klik "Hapus" jika ingin mengganti

### **2. Validasi:**
1. File akan divalidasi otomatis
2. Error message muncul jika tidak valid
3. File yang valid akan tampil preview
4. Form bisa disubmit setelah valid

### **3. Submit:**
1. Lengkapi semua field wajib
2. Upload foto KTP (opsional)
3. Klik "Simpan Data"
4. Data tersimpan dengan foto

---

## 🔧 **Konfigurasi**

### **PHP Settings:**
```php
// upload_max_filesize = 5M
// post_max_size = 10M
// max_file_uploads = 20
```

### **Folder Permissions:**
```bash
chmod 755 uploads/ktp/
```

### **Database:**
```sql
-- Kolom foto_ktp sudah ditambahkan
-- Index sudah dibuat untuk performa
```

---

## ✅ **Status: SELESAI**

**Fitur Upload Foto KTP telah berhasil ditambahkan!**

**Fitur yang tersedia:**
- ✅ Upload foto KTP dengan validasi
- ✅ Preview foto real-time
- ✅ Drag & drop support
- ✅ Validasi format dan ukuran
- ✅ Nama file otomatis unik
- ✅ Database integration
- ✅ UI/UX yang menarik

---

## 🔗 **Akses Demo**

- **Demo Form**: http://localhost:8080/demo_add_customer.php
- **Main Form**: http://localhost:8080/add_customer.php (login required)

**Form tambah data pelanggan sekarang dilengkapi dengan fitur upload foto KTP yang lengkap!** 🎉

**Developed by Amiruloh untuk Tugas Akhir D3 Manajemen Informatika AMIK CBI Sukabumi**



# 1. Install dependencies
cd /home/lab-mji/ensia1
composer install

# 2. Setup database
mysql -u root -p
CREATE DATABASE ensia_db;
CREATE USER 'ensia_user'@'localhost' IDENTIFIED BY 'ensia_password';
GRANT ALL PRIVILEGES ON ensia_db.* TO 'ensia_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;

# 3. Import database
mysql -u ensia_user -pensia_password ensia_db < database/schema.sql

# 4. Set permissions
chmod 755 uploads/ uploads/ktp/
mkdir -p /tmp/php_sessions && chmod 777 /tmp/php_sessions

# 5. Buat konfigurasi PHP
cat > php.ini << 'EOF'
[Session]
session.save_path = "/tmp/php_sessions"
session.gc_maxlifetime = 1440
session.cookie_lifetime = 0
session.use_cookies = 1
session.use_only_cookies = 1
session.cookie_httponly = 1
EOF

# 6. Jalankan server
php -S localhost:8080 -c php.ini

# 7. Buka browser
# http://localhost:8080